/*---------------------------------------------------------------------------*\

    FILE....: OSC.H
    TYPE....: C Module
    AUTHOR..: David Rowe
    DATE....: 6/11/97

    Digital ocillator function, TC simulation version.

\*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

         Voicetronix Voice Processing Board (VPB) Software

         Copyright (C) 1999-2001 Voicetronix www.voicetronix.com.au

         This library is free software; you can redistribute it and/or
         modify it under the terms of the GNU Lesser General Public
         License as published by the Free Software Foundation; either
         version 2.1 of the License, or (at your option) any later version.

         This library is distributed in the hope that it will be useful,
         but WITHOUT ANY WARRANTY; without even the implied warranty of
         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
         Lesser General Public License for more details.

         You should have received a copy of the GNU Lesser General Public
         License along with this library; if not, write to the Free Software
         Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
	 USA

\*---------------------------------------------------------------------------*/

#include <assert.h>
#include <stdio.h>
#include <math.h>
#include "osc.h"

/*--------------------------------------------------------------------------*\

				FUNCTIONS

\*--------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*\

	FUNCTION.: osc_init
	AUTHOR...: David Rowe
	DATE.....: 6/11/97

	Initialises the state variables of a digital oscillator.

\*--------------------------------------------------------------------------*/

void osc_init(OSC *osc, short freq, USHORT mag)
/*  OSC    *osc;	OSC state variable structure	*/
/*  short  freq;	2cos(w) in Q14			*/
/*  USHORT mag;		magnitude of osc, 0dBm0 = 32767	*/
{

    assert(osc != NULL);

    osc->freq = (float)freq/pow(2,14);
    osc->mag = mag;
    osc->y1 = mag;
    osc->y2 = mag*osc->freq/2.0;
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: osc_oscillate
	AUTHOR...: David Rowe
	DATE.....: 6/11/97

	Digital oscillator function, implemented as a marginally stable 2nd
	order IIR filter.  The ouput samples are summed with the exisiting
	contents of the buffer.

\*--------------------------------------------------------------------------*/

void osc_oscillate(OSC *osc, short buf[], USHORT n)
/*  OSC    *osc;	oscillator state variables		*/
/*  short  buf;		buffer of samples 			*/
/*  USHORT n;		size of buffer				*/
{
    float  c,acc;	/* 2cos(w), accumulator		        */
    float  y1,y2;	/* state variables			*/
    int    i;

    /* load state variables into locals */

    c = osc->freq;
    y1 = osc->y1;
    y2 = osc->y2;

    /*
       y(n) = 2*cos(w) * y(n-1) - y(n-2)
    */

    for(i=0; i<n; i++) {
	acc = c*y1 - y2;	    
	y2 = y1;
	y1 = acc;	
	buf[i] += acc;
    }

    /* load locals back into state variables */

    osc->y1 = y1;
    osc->y2 = y2;
}

